﻿// Copyright (c) Strange Loop Games. All rights reserved.
// See LICENSE file in the project root for full license information.

namespace Eco.Mods
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using Eco.Gameplay.Items;
    using Eco.Gameplay.Players;
    using Eco.Gameplay.Systems.Messaging.Chat.Commands;
    using Eco.Mods.TechTree;
    using Eco.Shared.Graphics;
    using Eco.Shared.Math;
    using Eco.Shared.Networking;
    using Eco.Shared.Utils;
    using Eco.World;
    using Eco.World.Color;
    using Eco.Gameplay.Rooms;
    using Eco.World.Blocks;

    [ChatCommandHandler]
    public class PaintingCommands
    {
        [ChatSubCommand("Inventory", "Give yourself mixed paint for predefined named colors", "paint", ChatAuthorizationLevel.Admin)]
        public static void GivePaint(User user, int namedColorIndex, int number = 1)
        {
            var color = ((NamedColors)namedColorIndex).GetNamedColor();
            GivePaintInternal(user, color, number);
        }
        
        [ChatSubCommand("Land", "Clears paint in area with radius specified around the player. Max = 20", ChatAuthorizationLevel.Admin)]
        public static void ClearPaint(User user, int radius = 4)
        {
            radius = Math.Clamp(radius, 1, 20); // prevent too big volume calls
            var center = user.Position.XYZi();
            var offset = new Vector3i(radius, radius, radius);
            var range = new WorldRange(center - offset, center + offset);

            // Clear color in batch and then trigger dirty struct update
            BlockColorManager.Obj.ClearColors(range.XYZIterInc(), true);

            // Client prediction as we do not want to force all chunk rebuild in case range is big,
            // just color changes locally and then lazy chunk updater will do its work
            user.Player?.RPC("ClearBlockPaint", center, radius);
        }
        
        [ChatSubCommand("Inventory", "Give yourself mixed paint with any rgb color", "paintrgb", ChatAuthorizationLevel.Admin)]
        public static void GivePaintRGB(User user, int r, int g, int b, int amount = 1)
        {
            GivePaintInternal(user, new ByteColor((byte)r, (byte)g, (byte)b, 255), amount);
        }
        
        [ChatSubCommand("Preset", "Lots of different paint buckets for test", ChatAuthorizationLevel.DevTier)]
        public static void Paint(User user, int stackCount = 5)
        {
            // Build random unique buckets list
            var paintBuckets = new List<PaintBucketItem>();
            for (var i = 0; i < stackCount; i++)
                paintBuckets.Add(GetPaint(NamedColorUtils.GetRandomNamedColorAll().Color));
            
            // Reuse presets core to spawn previously built list
            ItemPresetsCommands.SpawnStorage(user, "Paint buckets", 0, paintBuckets, 100);
        }
        
        [ChatSubCommand("Land", "Spawns giant walls and paints them with random colors", "painttest", ChatAuthorizationLevel.Admin)]
        public static void PaintTest(User user, int size = 50, int walls = 5)
        {
            var startPos = user.Position.XYZi();
            for (var w = 0; w < walls; w++)
            {
                var zOffset = 1 + (w * 3); // offset for walls on z axis
                
                for (var i = 0; i < size; i++)
                {
                    for (var j = 0; j < size; j++)
                    {
                        var namedColor = NamedColorUtils.GetRandomNamedColorAll();
                        var targetPos = startPos + new Vector3i(1 + j, 1 + i, zOffset);
                        World.SetBlock(typeof(HewnLogFloorBlock), targetPos);
                        BlockColorManager.Obj.SetColor(targetPos, namedColor.Color);
                    }
                }
            }
        }

        [ChatSubCommand("Room", "Test painted block housing value by checking current room", "testpaintedroom", ChatAuthorizationLevel.Admin)]
        public static void TestPaintedRoom(User user)
        {
            user.UpdateRoom();
            var room = user.CurrentRoom;
            
            if (room?.RoomStats?.Contained ?? false)
            {
                var paintedCount        = room.RoomStats.PaintedBlockCount;
                var paintedPercentage   = room.RoomStats.PaintedBlockPercentage;
                var config              = RoomData.Obj.RoomConfig.PaintedBlockHousingBonus;
                var totalValue          = room.RoomValue?.Value ?? 0f;
                var paintedContribution = paintedPercentage * config;
                
                user.Player?.MsgLocStr($"Room painted blocks: {paintedCount}/{room.RoomStats.PaintableBlockCount} ({paintedPercentage:P1})");
                user.Player?.MsgLocStr($"Painted block value (config): {config}");
                user.Player?.MsgLocStr($"Total room value: {totalValue}");
                user.Player?.MsgLocStr($"Painted block contribution: {paintedContribution:F3} ({paintedPercentage:P1} × {config})");
            }
            else
            {
                user.Player?.MsgLocStr("You are not in a valid room.");
            }
        }

        [ChatSubCommand("Room", "Paint all constructed blocks within radius to specified color", "paintroom", ChatAuthorizationLevel.Admin)]
        public static void PaintRoom(User user, string colorName, int radius = 10)
        {
            // Validate radius
            radius = Math.Clamp(radius, 1, 50);
            
            // Try to find the color by name
            NamedColor? foundColor = null;
            
            // First try exact match with base colors
            if (Enum.TryParse<NamedColors>(colorName, true, out var namedColor))
            {
                foundColor = new NamedColor(namedColor.ToString(), namedColor.GetNamedColor());
            }
            else
            {
                // Try to find in all available colors (including blends)
                var allColors = NamedColorUtils.AllColors;
                foundColor = allColors.FirstOrDefault(c => 
                    string.Equals(c.Name, colorName, StringComparison.OrdinalIgnoreCase) ||
                    string.Equals(c.GetDisplayName().NotTranslated, colorName, StringComparison.OrdinalIgnoreCase));
            }
            
            if (foundColor == null)
            {
                user.Player?.MsgLocStr($"Color '{colorName}' not found!");
                user.Player?.MsgLocStr("Available base colors:");
                
                // Show base colors
                var baseColorNames = Enum.GetNames<NamedColors>().Where(name => name != "White" || name == "White").OrderBy(x => x);
                user.Player?.MsgLocStr($"  {string.Join(", ", baseColorNames)}");
                
                user.Player?.MsgLocStr("You can also use blend colors like 'Red #001', 'Blue #012', etc.");
                user.Player?.MsgLocStr("Use exact color names (case insensitive).");
                return;
            }
            
            var targetColor = foundColor.Value.Color;
            var userPos = user.Position.XYZi();
            var paintedCount = 0;
            
            // Search in a cube around the user
            for (int x = -radius; x <= radius; x++)
            {
                for (int y = -radius; y <= radius; y++)
                {
                    for (int z = -radius; z <= radius; z++)
                    {
                        var targetPos = userPos + new Vector3i(x, y, z);
                        var block = World.GetBlock(targetPos);
                        
                        // Check if it's a constructed block that can be painted
                        if (block != null && block.Is<Constructed>() && block is IColoredBlock)
                        {
                            // Set the paint color
                            BlockColorManager.Obj.SetColor(targetPos, targetColor);
                            paintedCount++;
                        }
                    }
                }
            }
            
            user.Player?.MsgLocStr($"Painted {paintedCount} constructed blocks with '{foundColor.Value.GetDisplayName().NotTranslated}' within radius {radius}");
        }

        static PaintBucketItem GetPaint(ByteColor color)
        {
            // Creating unique paint item.
            var mixedPaint = Item.Create<PaintBucketItem>();
            mixedPaint.SetColor(color);
            return mixedPaint;
        }

        static void GivePaintInternal(User user, ByteColor color, int amount = 1)
        {
            using var changeSet = InventoryChangeSet.New(user.Inventory.ToolbarBackpack, user);
            
            // Creating unique paint item.
            var mixedPaint = GetPaint(color);
            
            // Todo works badly with item comparison for uniques, need to fix
            changeSet.AddItem(mixedPaint, amount);
            var res = changeSet.TryApply();
            if (!res.Success) user.MsgLocStr(res.Message);
        }
    }
}
