﻿// Copyright (c) Strange Loop Games. All rights reserved.
// See LICENSE file in the project root for full license information.

namespace Eco.Mods
{
    using Eco.Core.Tests;
    using Eco.Gameplay.Players;
    using Eco.Gameplay.Systems.Messaging.Notifications;
    using Eco.Gameplay.Systems.Messaging.Chat.Commands;
    using Eco.Shared.Localization;
    using Eco.Shared.Utils;
    using Eco.Simulation.WorldLayers;
    using Eco.Simulation.WorldLayers.Layers;
    using Eco.Simulation.Animals;
    using Eco.Simulation.Types;
    using Eco.Shared.Services;
    using Eco.Simulation.Agents;
    using Eco.Shared.Networking;
    using Eco.Shared.Math;
    using System;
    using System.Linq;
    using System.Collections.Generic;
    using Eco.Simulation;
    using Eco.Simulation.Settings;

    [ChatCommandHandler]
    public static class SimCommands
    {
        /// <summary>Queue to store population changes during a world tick</summary>
        private static readonly List<PopulationChange> PopulationChangesQueue = new List<PopulationChange>();

        /// <summary>Last processed population changes for display purposes</summary>
        private static readonly List<PopulationChange> LastPopulationChanges = new List<PopulationChange>();

        /// <summary>Whether animal population change notifications are enabled</summary>
        private static bool animalPopulationNotificationsEnabled = false;

        /// <summary>Structure to hold population change data</summary>
        private struct PopulationChange
        {
            public string LayerName;
            public float PopulationBefore;
            public float PopulationAfter;
            public float TotalRounded;
            public float Change => this.PopulationAfter - this.PopulationBefore;
        }

        // Subscribe to animal population changes when the class is first loaded
        static SimCommands()
        {
            AnimalLayer.PopulationTickChangedEvent.Add(OnAnimalPopulationChanged);
            AnimalsManager.AnimalPopulationReducedEvent.Add(OnAnimalPopulationReduced);
            AnimalsManager.SpeciesWentExtinctEvent.Add((species, source) => 
            {
                var message = source switch
                {
                    Player player => Localizer.Do($"EXTINCTION: {species.DisplayName} has gone extinct! The last one was killed by {player.User.MarkedUpName}."),
                    _ => Localizer.Do($"EXTINCTION: {species.DisplayName} has gone extinct!")
                };
                NotificationManager.ServerMessageToAll(message, category: NotificationCategory.Notifications, style: NotificationStyle.Chat);
            });
            AnimalsManager.SpeciesRecoveredEvent.Add(species => NotificationManager.ServerMessageToAll(Localizer.Do($"RECOVERY: {species.DisplayName} has recovered from extinction!"), category: NotificationCategory.Notifications, style: NotificationStyle.Chat));
            
            // Subscribe to world tick finished event to process queued population changes
            WorldLayerManager.Obj.OnTicked.Add(ProcessQueuedPopulationChanges);
        }

        /// <summary>
        /// Handler for when animal population is reduced due to hunting - displays notification if enabled
        /// </summary>
        private static void OnAnimalPopulationReduced(Animal animal, INetObject source, Vector2i worldPos, float oldPopulation, float newPopulation)
        {
            if (!animalPopulationNotificationsEnabled) return;

            string hunterName = source switch
            {
                Player player => player.DisplayName,
                _ => "Something"
            };

            var message = Localizer.Do($"{hunterName} killed a {animal.Species.DisplayName}. Population reduced from {oldPopulation.StyledNum()} to {newPopulation.StyledNum()} at {worldPos}");
            
            NotificationManager.ServerMessageToAll(message, 
                category: NotificationCategory.Notifications, 
                style: NotificationStyle.Chat, 
                forceTemporary: true);
        }

        /// <summary>Handler for animal population changes - queues changes instead of sending immediate notifications</summary>
        private static void OnAnimalPopulationChanged(string layerName, float populationBefore, float populationAfter)
        {
            if (!animalPopulationNotificationsEnabled) return;

            var layer = WorldLayerManager.Obj.GetLayer(layerName);
            var totalRounded = layer?.RoundedTotal ?? 0f;

            lock (PopulationChangesQueue)
            {
                PopulationChangesQueue.Add(new PopulationChange
                {
                    LayerName = layerName,
                    PopulationBefore = populationBefore,
                    PopulationAfter = populationAfter,
                    TotalRounded = totalRounded
                });
            }
        }

        /// <summary>Formats population changes with hourly and daily rate estimates</summary>
        private static List<string> FormatPopulationChanges(List<PopulationChange> changes)
        {
            // Calculate rates based on world tick timing
            var tickIntervalSeconds = EcoDef.DefaultWorldLayerTickTime; // Default is 10 minutes (600 seconds)
            var ticksPerHour = 3600.0 / tickIntervalSeconds; // Ticks per hour
            var ticksPerDay = 86400.0 / tickIntervalSeconds; // Ticks per day

            return changes
                .Select(c => 
                {
                    var changeText = c.Change > 0 ? $"+{c.Change.StyledNum()}" : c.Change.StyledNum();
                    var hourlyRate = c.Change * (float)ticksPerHour;
                    var dailyRate = c.Change * (float)ticksPerDay;
                    
                    var hourlyText = hourlyRate > 0 ? $"+{hourlyRate.StyledNum()}" : hourlyRate.StyledNum();
                    var dailyText = dailyRate > 0 ? $"+{dailyRate.StyledNum()}" : dailyRate.StyledNum();
                    
                    // Get habitability total if this is an animal layer
                    var animalLayer = WorldLayerManager.Obj.GetLayer(c.LayerName) as AnimalLayer;
                    var habitabilityText = string.Empty;
                    if (animalLayer?.Settings is AnimalLayerSettings animalSettings && !string.IsNullOrEmpty(animalSettings.HabitabilityLayer))
                    {
                        var habitabilityLayer = WorldLayerManager.Obj.GetLayer(animalSettings.HabitabilityLayer);
                        if (habitabilityLayer != null && animalLayer.Settings != null)
                        {
                            var maxPossible = habitabilityLayer.ActualTotal * animalLayer.Settings.Range.Max;
                            habitabilityText = $"    Total Possible: {maxPossible.StyledNum()}";
                        }
                    }
                    
                    return Localizer.Do($"{c.LayerName}: {c.PopulationBefore.StyledNum()} → {c.PopulationAfter.StyledNum()} ({c.TotalRounded} rounded) ({changeText})    Current rate per hour: {hourlyText}    Current rate per day: {dailyText}{habitabilityText}").ToString();
                })
                .ToList();
        }

        /// <summary>Processes queued population changes and sends a single consolidated notification</summary>
        private static void ProcessQueuedPopulationChanges()
        {
            List<PopulationChange> changes;
            lock (PopulationChangesQueue)
            {
                if (PopulationChangesQueue.Count == 0) return;
                
                changes = new List<PopulationChange>(PopulationChangesQueue);
                PopulationChangesQueue.Clear();
            }

            // Filter changes > 0.01 and sort by change descending (positive changes first)
            var significantChanges = changes
                .Where(c => Math.Abs(c.Change) > 0.01f)
                .OrderByDescending(c => c.Change)
                .ToList();

            // Store the last processed changes for display purposes
            lock (LastPopulationChanges)
            {
                LastPopulationChanges.Clear();
                LastPopulationChanges.AddRange(significantChanges);
            }

            if (!significantChanges.Any() || !animalPopulationNotificationsEnabled) return;

            var changeLines = FormatPopulationChanges(significantChanges);
            var message = Localizer.Do($"Population Changes:\n{string.Join("\n", changeLines)}");
            
            NotificationManager.ServerMessageToAll(message, 
                category: NotificationCategory.Notifications, 
                style: NotificationStyle.Chat, 
                forceTemporary: true);
        }

        [ChatSubCommand("Sim", "Shows the last set of cached population changes with hourly and daily rate estimates", "popchanges", ChatAuthorizationLevel.User)]
        public static void ShowPopulationChanges(User user)
        {
            List<PopulationChange> lastChanges;
            lock (LastPopulationChanges)
            {
                if (LastPopulationChanges.Count == 0)
                {
                    NotificationManager.ServerMessageToPlayer(Localizer.DoStr("No population changes recorded yet."), user, style: NotificationStyle.Chat);
                    return;
                }
                lastChanges = new List<PopulationChange>(LastPopulationChanges);
            }

            var changeLines = FormatPopulationChanges(lastChanges);
            var tickIntervalMinutes = EcoDef.DefaultWorldLayerTickTime / 60;
            var message = Localizer.Do($"Last Population Changes (tick interval: {tickIntervalMinutes} min):\n{string.Join("\n", changeLines)}");
            
            NotificationManager.ServerMessageToPlayer(message, user, style: NotificationStyle.Chat);
        }

        [ChatSubCommand("Sim", "Lists animal populations at specified location (or current location), including animals with 0 population", "animalpop", ChatAuthorizationLevel.User)]
        public static void ListAnimalPopulations(User user, float x = -1, float z = -1)
        {
            Vector2i worldPos;
            
            // If coordinates not provided, use user's current position
            if (x == -1 && z == -1)
                worldPos = user.Position.XZi();
            else
                worldPos = new Vector2i((int)x, (int)z);

            // Get all animal species and check their populations, then sort by population desc
            var populationData = EcoSim.AllSpecies.OfType<AnimalSpecies>()
                .Select(species => new { 
                    Species = species, 
                    Layer = species.AnimalLayer,
                    LocalPopulation = species.AnimalLayer?.SafeEntry(species.AnimalLayer.WorldPosToLayerPos(worldPos)) ?? 0f,
                    TotalPopulation = species.AnimalLayer?.RoundedTotal ?? 0
                })
                .Where(x => x.Layer != null)
                .OrderByDescending(x => x.LocalPopulation)
                .ToList();
            
            var populationInfo = populationData
                .Select(x => $"{x.Species.DisplayName}: {x.LocalPopulation.StyledNum()} / {x.TotalPopulation.StyledNum()}")
                .ToList();
            
            if (populationInfo.Any())
            {
                var message = Localizer.Do($"Animal populations at {worldPos}:\n{string.Join("\n", populationInfo)}");
                NotificationManager.ServerMessageToPlayer(message, user, style: NotificationStyle.Chat); 
            }
            else
                NotificationManager.ServerMessageToPlayer(Localizer.Do($"No animals found at location {worldPos}."), user, style: NotificationStyle.Chat);
        }
        
        [ChatSubCommand("Sim", "Lists plant populations at specified location (or current location), including underwater plants and plants with 0 population", "plantpop", ChatAuthorizationLevel.User)]
        public static void ListPlantPopulations(User user, float x = -1, float z = -1)
        {
            Vector2i worldPos;
            
            // If coordinates not provided, use user's current position
            if (x == -1 && z == -1)
                worldPos = user.Position.XZi();
            else
                worldPos = new Vector2i((int)x, (int)z);

            // Get all plant species and check their populations, then sort by population desc
            var populationData = EcoSim.AllSpecies.OfType<PlantSpecies>()
                .Select(species => {
                    var layer = species.Layer as PlantLayer;
                    return new {
                        Species = species,
                        Layer = layer,
                        LocalPopulation = layer?.SafeEntry(layer.WorldPosToLayerPos(worldPos)) ?? 0f,
                        TotalPopulation = layer?.RoundedTotal ?? 0
                    };
                })
                .Where(x => x.Layer != null)
                .OrderByDescending(x => x.LocalPopulation)
                .ToList();
            
            var populationInfo = populationData
                .Select(x => $"{x.Species.DisplayName}: {x.LocalPopulation.StyledNum()} / {x.TotalPopulation.StyledNum()}")
                .ToList();
            
            if (populationInfo.Any())
            {
                var message = Localizer.Do($"Plant populations at {worldPos}:\n{string.Join("\n", populationInfo)}");
                NotificationManager.ServerMessageToPlayer(message, user, style: NotificationStyle.Chat); 
            }
            else
                NotificationManager.ServerMessageToPlayer(Localizer.Do($"No plants found at location {worldPos}."), user, style: NotificationStyle.Chat);
        }

        [ChatSubCommand("Sim", "Toggles animal population change notifications on/off to all users.", "animalnotify", ChatAuthorizationLevel.Admin)]
        public static void ToggleAnimalNotifications(User user)
        {
            animalPopulationNotificationsEnabled = !animalPopulationNotificationsEnabled;
            var status = animalPopulationNotificationsEnabled ? "enabled" : "disabled";
            NotificationManager.ServerMessageToPlayer(
                Localizer.Do($"Animal population notifications {status}."), 
                user, 
                style: NotificationStyle.InfoBox);
        }

        [ChatSubCommand("Sim", "Destroys a specified amount of a given species from surrounding area", "cull", ChatAuthorizationLevel.Admin)]
        public static void CullSpecies(User user, string speciesName, float amount = 10f, float distance = 50f)
        {
            // Find the species by name (case-insensitive partial match)
            var species = EcoSim.AllSpecies.OfType<AnimalSpecies>()
                .FirstOrDefault(s => s.DisplayName.ToString().ToLower().Contains(speciesName.ToLower()));
            
            if (species == null)
            {
                NotificationManager.ServerMessageToPlayer(
                    Localizer.Do($"Species '{speciesName}' not found."), 
                    user, 
                    style: NotificationStyle.Chat);
                return;
            }

            if (species.AnimalLayer == null)
            {
                NotificationManager.ServerMessageToPlayer(
                    Localizer.Do($"Species '{species.DisplayName}' has no animal layer."), 
                    user, 
                    style: NotificationStyle.Chat);
                return;
            }

            var layer = species.AnimalLayer;
            var userWorldPos = user.Position.XZi();
            var centerLayerPos = layer.WorldPosToLayerPos(userWorldPos);
            
            float totalRemoved = 0f;
            float totalBefore = layer.RoundedTotal;
            var cellsAffected = 0;
            
            // Start with specified distance and expand if needed
            float currentSearchDistance = distance;
            float maxSearchDistance = currentSearchDistance;
            var processedCells = new HashSet<Vector2i>();
            
            // Keep expanding search until we find enough animals or exhaust possibilities
            while (totalRemoved < amount)
            {
                var layerRadius = (int)Math.Ceiling(currentSearchDistance / layer.Settings.VoxelsPerEntry);
                var foundAnyInThisExpansion = false;
                
                // Search all cells within current distance
                for (int x = -layerRadius; x <= layerRadius; x++)
                {
                    for (int z = -layerRadius; z <= layerRadius; z++)
                    {
                        var layerPos = centerLayerPos + new Vector2i(x, z);
                        
                        // Skip if we've already processed this cell
                        if (processedCells.Contains(layerPos)) continue;
                        
                        var worldPos = layer.LayerPosToWorldPos(layerPos);
                        var actualDistance = (userWorldPos - worldPos).Length;
                        
                        // Only process cells within current search distance
                        if (actualDistance > currentSearchDistance) continue;
                        
                        // Mark as processed
                        processedCells.Add(layerPos);
                        
                        var currentPop = layer.SafeEntry(layerPos);
                        if (currentPop <= 0) continue;
                        
                        foundAnyInThisExpansion = true;
                        var toRemove = Math.Min(currentPop, amount - totalRemoved);
                        if (toRemove <= 0) break;
                        
                        layer.UpdateAtWorldPos(worldPos, (pos, val) => Math.Max(0f, val - toRemove));
                        
                        totalRemoved += toRemove;
                        cellsAffected++;
                        maxSearchDistance = Math.Max(maxSearchDistance, actualDistance);
                        
                        if (totalRemoved >= amount) break;
                    }
                    if (totalRemoved >= amount) break;
                }
                
                // If we've removed enough or found no new animals, stop
                if (totalRemoved >= amount || !foundAnyInThisExpansion) break;
                
                // Expand search distance
                currentSearchDistance += distance;
                
                // Safety check to prevent infinite expansion (max 10x original distance)
                if (currentSearchDistance > distance * 10) break;
            }

            // Mark layer as modified and recalculate totals
            WorldLayerSync.UpdateLayerTotals(layer);
            
            float totalAfter = layer.RoundedTotal;
            
            // Trigger population change event to enable extinction detection
            if (totalBefore != totalAfter)
                AnimalLayer.PopulationTickChangedEvent?.Invoke(layer.Settings.Name, totalBefore, totalAfter);
            
            var message = Localizer.Do($"Species Cull Summary:\nSpecies: {species.DisplayName}\nTarget amount: {amount.StyledNum()}\nSearch distance: {distance.StyledNum()} meters (expanded to {maxSearchDistance.StyledNum()} meters)\nActually removed: {totalRemoved.StyledNum()} from {cellsAffected} cells\nPopulation before: {totalBefore.StyledNum()}\nPopulation after: {totalAfter.StyledNum()}\nTotal population remaining: {totalAfter.StyledNum()}");
            
            NotificationManager.ServerMessageToPlayer(message, user, style: NotificationStyle.Chat);
        }

        [CITest]
        [ChatSubCommand("Sim", "Raises the sea level by a passed in amount.  Careful with this one!", ChatAuthorizationLevel.DevTier)]
        public static void RaiseSeaLevel(User user, float val = 1.5f)
        {
            var seaLevel = WorldLayerManager.Obj.ClimateSim.State.SeaLevel;
            WorldLayerManager.Obj.ClimateSim.SetSeaLevel(seaLevel + val);
            NotificationManager.ServerMessageToAll(Localizer.Format("{0} has raised the seas by {1}!", user.Name, Text.StyledNum(val)));
        }

        [CITest]
        [ChatSubCommand("Sim", "Displays the current sea level and how much it has risen.", "sea", ChatAuthorizationLevel.User)]
        public static void SeaLevel(User user) 
        {
            NotificationManager.TemporaryServerMessageToPlayer(Localizer.Format("Current sea level: {0}  Amount raised so far: {1}", Text.StyledNum(WorldLayerManager.Obj.ClimateSim.State.SeaLevel), Text.StyledNum(WorldLayerManager.Obj.ClimateSim.State.SeaLevel - WorldLayerManager.Obj.ClimateSim.State.InitialSeaLevel)), user);
        }

        [CITest]
        [ChatSubCommand("Sim", "Forces a world layer simulation tick to update layers immediately.", "wtick", ChatAuthorizationLevel.Admin)]
        public static void ForceWorldTick(User user)
        {
            WorldLayerManager.Obj.ForceTick();
            NotificationManager.TemporaryServerMessageToPlayer(Localizer.DoStr("Forced world layer tick executed."), user);
        }
    }
}
