/*
 * Decompiled with CFR 0.152.
 */
package org.hjson;

import java.io.IOException;
import java.io.Writer;
import java.util.regex.Pattern;
import org.hjson.HjsonDsf;
import org.hjson.HjsonOptions;
import org.hjson.HjsonParser;
import org.hjson.IHjsonDsfProvider;
import org.hjson.JsonArray;
import org.hjson.JsonObject;
import org.hjson.JsonValue;
import org.hjson.JsonWriter;

class HjsonWriter {
    private IHjsonDsfProvider[] dsfProviders;
    private boolean outputComments;
    private boolean bracesSameLine;
    private boolean allowCondense;
    private boolean allowMultiVal;
    private boolean emitRootBraces;
    private String space;
    private String commentSpace;
    static Pattern needsEscapeName = Pattern.compile("[,\\{\\[\\}\\]\\s:#\"']|//|/\\*");

    public HjsonWriter(HjsonOptions options) {
        if (options != null) {
            this.dsfProviders = options.getDsfProviders();
            this.bracesSameLine = options.bracesSameLine();
            this.allowCondense = options.getAllowCondense();
            this.allowMultiVal = options.getAllowMultiVal();
            this.space = options.getSpace();
            this.commentSpace = options.getCommentSpace();
            this.outputComments = options.getOutputComments();
            this.emitRootBraces = options.getEmitRootBraces();
        } else {
            this.dsfProviders = new IHjsonDsfProvider[0];
            this.bracesSameLine = false;
            this.allowCondense = true;
            this.allowMultiVal = true;
            this.emitRootBraces = false;
            this.space = "  ";
            this.commentSpace = "";
            this.outputComments = false;
        }
    }

    public HjsonWriter(HjsonOptions options, boolean outputComments) {
        this(options);
        this.outputComments = outputComments;
    }

    void nl(Writer tw, int level) throws IOException {
        tw.write(JsonValue.eol);
        this.indent(tw, level);
    }

    void indent(Writer tw, int level) throws IOException {
        for (int i = 0; i < level; ++i) {
            tw.write(this.space);
        }
    }

    void indentComment(Writer tw) throws IOException {
        tw.write(this.commentSpace);
    }

    public void save(JsonValue value, Writer tw, int level, String separator, boolean noIndent) throws IOException {
        this.save(value, tw, level, separator, noIndent, false);
    }

    public void save(JsonValue value, Writer tw, int level, String separator, boolean noIndent, boolean forceQuotes) throws IOException {
        if (value == null) {
            tw.write(separator);
            tw.write("null");
            return;
        }
        String dsfValue = HjsonDsf.stringify(this.dsfProviders, value);
        if (dsfValue != null) {
            tw.write(separator);
            tw.write(dsfValue);
            return;
        }
        if (this.outputComments && level == 0 && value.hasBOLComment()) {
            this.writeHeader(tw, value, level);
        }
        switch (value.getType()) {
            case OBJECT: {
                JsonObject obj = value.asObject();
                this.writeObject(obj, tw, level, separator, noIndent);
                break;
            }
            case ARRAY: {
                JsonArray arr = value.asArray();
                this.writeArray(arr, tw, level, separator, noIndent);
                break;
            }
            case BOOLEAN: {
                tw.write(separator);
                tw.write(value.isTrue() ? "true" : "false");
                break;
            }
            case STRING: {
                tw.write(separator);
                this.writeString(value.asString(), tw, level, forceQuotes, separator);
                break;
            }
            default: {
                tw.write(separator);
                if (forceQuotes) {
                    tw.write(34);
                }
                tw.write(value.toString());
                if (!forceQuotes) break;
                tw.write(34);
            }
        }
        if (this.outputComments && value.hasEOLComment()) {
            this.writeEOLComment(tw, value, level);
        }
    }

    void writeObject(JsonObject obj, Writer tw, int level, String separator, boolean noIndent) throws IOException {
        boolean emitBraces = this.emitBraces(obj, level);
        if (!emitBraces) {
            this.openContainer(tw, noIndent, obj.isCondensed(), level, separator, '{');
        }
        int index = 0;
        for (JsonObject.Member pair : obj) {
            if (this.outputComments && pair.getValue().hasBOLComment()) {
                this.writeBOLComment(tw, pair.getValue(), level);
            }
            this.handleContainerLines(tw, obj.isCondensed(), index, level, obj.getLineLength());
            tw.write(HjsonWriter.escapeName(pair.getName(), HjsonWriter.forceQuoteObject(obj)));
            tw.write(":");
            boolean forceQuoteValue = HjsonWriter.forceQuoteValue(pair.getValue(), obj, this.outputComments);
            this.save(pair.getValue(), tw, level + 1, " ", false, forceQuoteValue);
            ++index;
        }
        if (this.outputComments && obj.hasInteriorComment()) {
            this.writeInteriorComment(tw, obj, level);
        }
        if (!emitBraces) {
            this.closeContainer(tw, obj.isCondensed(), obj.size(), level, '}');
        }
    }

    void writeArray(JsonArray arr, Writer tw, int level, String separator, boolean noIndent) throws IOException {
        this.openContainer(tw, noIndent, arr.isCondensed(), level, separator, '[');
        int n = arr.size();
        for (int i = 0; i < n; ++i) {
            JsonValue element = arr.get(i);
            if (this.outputComments && element.hasBOLComment()) {
                this.writeBOLComment(tw, element, level);
            }
            this.handleContainerLines(tw, arr.isCondensed(), i, level, arr.getLineLength());
            boolean forceQuoteArray = HjsonWriter.forceQuoteArray(element, arr, this.outputComments);
            this.save(element, tw, level + 1, "", true, forceQuoteArray);
        }
        if (this.outputComments && arr.hasInteriorComment()) {
            this.writeInteriorComment(tw, arr, level);
        }
        this.closeContainer(tw, arr.isCondensed(), n, level, ']');
    }

    boolean emitBraces(JsonObject obj, int level) {
        return this.emitRootBraces && level == 0 && (!obj.hasBOLComment() || !this.outputComments);
    }

    void openContainer(Writer tw, boolean noIndent, boolean condensed, int level, String separator, char openWith) throws IOException {
        if (!noIndent) {
            if (this.bracesSameLine || condensed) {
                tw.write(separator);
            } else {
                this.nl(tw, level);
            }
        }
        tw.write(openWith);
    }

    void writeHeader(Writer tw, JsonValue value, int level) throws IOException {
        this.writeComment(value.getBOLComment(), tw, level);
        this.nl(tw, level);
    }

    void writeBOLComment(Writer tw, JsonValue value, int level) throws IOException {
        this.nl(tw, level + 1);
        this.writeComment(value.getBOLComment(), tw, level + 1);
    }

    void writeInteriorComment(Writer tw, JsonValue value, int level) throws IOException {
        this.nl(tw, level + 1);
        this.writeComment(value.getInteriorComment(), tw, level + 1);
    }

    void writeEOLComment(Writer tw, JsonValue value, int level) throws IOException {
        if (level == 0) {
            this.nl(tw, level);
            this.writeComment(value.getEOLComment(), tw, level);
        } else {
            tw.write(32);
            tw.write(value.getEOLComment());
        }
    }

    void handleContainerLines(Writer tw, boolean compact, int index, int level, int lineLength) throws IOException {
        if (!this.allowMultiVal) {
            this.nl(tw, level + 1);
        } else if (index % lineLength == 0) {
            if (!compact || !this.allowCondense) {
                this.nl(tw, level + 1);
            } else if (index > 0) {
                tw.write(", ");
            } else {
                tw.write(32);
            }
        } else {
            tw.write(", ");
        }
    }

    void closeContainer(Writer tw, boolean compact, int size, int level, char closeWith) throws IOException {
        if (size > 0) {
            if (compact && this.allowCondense && this.allowMultiVal) {
                tw.write(32);
            } else {
                this.nl(tw, level);
            }
        }
        tw.write(closeWith);
    }

    static String escapeName(String name) {
        return HjsonWriter.escapeName(name, false);
    }

    static String escapeName(String name, boolean force) {
        if (force || name.length() == 0 || needsEscapeName.matcher(name).find()) {
            return "\"" + JsonWriter.escapeString(name) + "\"";
        }
        return name;
    }

    void writeString(String value, Writer tw, int level, boolean forceQuotes, String separator) throws IOException {
        char[] valuec;
        if (value.length() == 0) {
            tw.write("\"\"");
            return;
        }
        char left = value.charAt(0);
        char right = value.charAt(value.length() - 1);
        char left1 = value.length() > 1 ? value.charAt(1) : (char)'\u0000';
        char left2 = value.length() > 2 ? value.charAt(2) : (char)'\u0000';
        boolean doEscape = false;
        for (char ch : valuec = value.toCharArray()) {
            if (!HjsonWriter.needsQuotes(ch)) continue;
            doEscape = true;
            break;
        }
        if (doEscape || HjsonParser.isWhiteSpace(left) || HjsonParser.isWhiteSpace(right) || left == '\"' || left == '\'' || left == '#' || left == '/' && (left1 == '*' || left1 == '/') || JsonValue.isPunctuatorChar(left) || HjsonParser.tryParseNumber(value, true) != null || HjsonWriter.startsWithKeyword(value)) {
            boolean noEscape = true;
            for (char ch : valuec) {
                if (!HjsonWriter.needsEscape(ch)) continue;
                noEscape = false;
                break;
            }
            if (noEscape) {
                tw.write("\"" + value + "\"");
                return;
            }
            boolean noEscapeML = true;
            boolean allWhite = true;
            for (char ch : valuec) {
                if (HjsonWriter.needsEscapeML(ch)) {
                    noEscapeML = false;
                    break;
                }
                if (HjsonParser.isWhiteSpace(ch)) continue;
                allWhite = false;
            }
            if (noEscapeML && !allWhite && !value.contains("'''")) {
                this.writeMLString(value, tw, level);
            } else {
                tw.write("\"" + JsonWriter.escapeString(value) + "\"");
            }
        } else {
            if (forceQuotes) {
                tw.write(34);
            }
            tw.write(value);
            if (forceQuotes) {
                tw.write(34);
            }
        }
    }

    void writeMLString(String value, Writer tw, int level) throws IOException {
        String[] lines = value.replace("\r", "").split("\n", -1);
        if (lines.length == 1) {
            tw.write("'''");
            tw.write(lines[0]);
            tw.write("'''");
        } else {
            this.nl(tw, ++level);
            tw.write("'''");
            for (String line : lines) {
                this.nl(tw, line.length() > 0 ? level : 0);
                tw.write(line);
            }
            this.nl(tw, level);
            tw.write("'''");
        }
    }

    void writeComment(String comment, Writer tw, int level) throws IOException {
        String[] lines = comment.split("\r?\n");
        this.indentComment(tw);
        tw.write(lines[0]);
        for (int i = 1; i < lines.length; ++i) {
            this.nl(tw, level);
            this.indentComment(tw);
            tw.write(lines[i]);
        }
    }

    static boolean startsWithKeyword(String text) {
        int p;
        if (text.startsWith("true") || text.startsWith("null")) {
            p = 4;
        } else if (text.startsWith("false")) {
            p = 5;
        } else {
            return false;
        }
        while (p < text.length() && HjsonParser.isWhiteSpace(text.charAt(p))) {
            ++p;
        }
        if (p == text.length()) {
            return true;
        }
        char ch = text.charAt(p);
        return ch == ',' || ch == '}' || ch == ']' || ch == '#' || ch == '/' && text.length() > p + 1 && (text.charAt(p + 1) == '/' || text.charAt(p + 1) == '*');
    }

    static boolean forceQuoteArray(JsonValue value, JsonArray array, boolean outputComments) {
        return value.isString() && (array.isCondensed() || array.getLineLength() > 1 || outputComments && value.hasEOLComment());
    }

    static boolean forceQuoteValue(JsonValue value, JsonObject object, boolean outputComments) {
        return value.isString() && (object.isCondensed() || object.getLineLength() > 1 || outputComments && value.hasEOLComment());
    }

    static boolean forceQuoteObject(JsonObject object) {
        return object.isCondensed() || object.getLineLength() > 1;
    }

    static boolean needsQuotes(char c) {
        switch (c) {
            case '\b': 
            case '\t': 
            case '\n': 
            case '\f': 
            case '\r': {
                return true;
            }
        }
        return false;
    }

    static boolean needsEscape(char c) {
        switch (c) {
            case '\"': 
            case '\\': {
                return true;
            }
        }
        return HjsonWriter.needsQuotes(c);
    }

    static boolean needsEscapeML(char c) {
        switch (c) {
            case '\t': 
            case '\n': 
            case '\r': {
                return false;
            }
        }
        return HjsonWriter.needsQuotes(c);
    }
}

